<?php

class Helpers
{
  public static function extractMapSrc($input)
  {
    if (empty($input))
      return null;

    // If it's a full iframe tag
    if (preg_match('/src="([^"]+)"/', $input, $matches)) {
      return $matches[1];
    }

    // If it's already a URL, return it (basic validation)
    if (filter_var($input, FILTER_VALIDATE_URL)) {
      // Check if it's a google map embed URL roughly
      if (strpos($input, 'google.com/maps/embed') !== false) {
        return $input;
      }
    }

    return $input;
  }

  public static function formatDate($dateString)
  {
    if (empty($dateString))
      return '';
    setlocale(LC_TIME, 'pt_BR', 'pt_BR.utf-8', 'portuguese');
    $timestamp = strtotime($dateString);
    // Fallback array in case setlocale doesn't work on Windows/Server
    $months = [
      'Jan' => 'Jan',
      'Feb' => 'Fev',
      'Mar' => 'Mar',
      'Apr' => 'Abr',
      'May' => 'Mai',
      'Jun' => 'Jun',
      'Jul' => 'Jul',
      'Aug' => 'Ago',
      'Sep' => 'Set',
      'Oct' => 'Out',
      'Nov' => 'Nov',
      'Dec' => 'Dez',
      'January' => 'Janeiro',
      'February' => 'Fevereiro',
      'March' => 'Março',
      'April' => 'Abril',
      'May' => 'Maio',
      'June' => 'Junho',
      'July' => 'Julho',
      'August' => 'Agosto',
      'September' => 'Setembro',
      'October' => 'Outubro',
      'November' => 'Novembro',
      'December' => 'Dezembro'
    ];

    $day = date('d', $timestamp);
    $month = date('F', $timestamp);
    $year = date('Y', $timestamp);

    $monthPt = $months[$month] ?? $month;

    return "{$day} de {$monthPt}, {$year}";
  }

  public static function uploadFile($file, $folder = 'uploads')
  {
    if (!isset($file['error']) || is_array($file['error'])) {
      return null; // Invalid upload
    }

    // Check for errors
    if ($file['error'] !== UPLOAD_ERR_OK) {
      return null;
    }

    $allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
    $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));

    if (!in_array($extension, $allowedExtensions)) {
      return null; // Invalid extension
    }

    // Create folder if not exists
    $targetDir = __DIR__ . '/../../public/' . $folder;
    if (!is_dir($targetDir)) {
      mkdir($targetDir, 0777, true);
    }

    // Simple unique name: timestamp_random.ext
    $newFilename = time() . '_' . rand(1000, 9999) . '.' . $extension;
    $targetPath = $targetDir . '/' . $newFilename;

    if (move_uploaded_file($file['tmp_name'], $targetPath)) {
      return '/' . $folder . '/' . $newFilename;
    }

    return null;
  }

  public static function sendEmail($to, $subject, $body, $isHtml = true)
  {
    // Load Settings from DB
    $smtpConfig = [
      'host' => 'smtp.example.com',
      'port' => 587,
      'user' => 'user@example.com',
      'pass' => 'secret',
      'from' => 'noreply@cevai.com',
      'name' => 'Cê Vai Tickets'
    ];

    // Try to fetch real settings
    try {
      if (file_exists(APP_PATH . '/models/Setting.php')) {
        require_once APP_PATH . '/models/Setting.php';
        if (class_exists('Setting')) {
          $settingModel = new Setting();
          $dbSettings = $settingModel->getAllAsArray();

          if (!empty($dbSettings['smtp_host']))
            $smtpConfig['host'] = $dbSettings['smtp_host'];
          if (!empty($dbSettings['smtp_port']))
            $smtpConfig['port'] = intval($dbSettings['smtp_port']);
          if (!empty($dbSettings['smtp_user']))
            $smtpConfig['user'] = $dbSettings['smtp_user'];
          if (!empty($dbSettings['smtp_pass']))
            $smtpConfig['pass'] = $dbSettings['smtp_pass'];
          // Use site name as sender name if available
          if (!empty($dbSettings['site_name']))
            $smtpConfig['name'] = $dbSettings['site_name'];
          // Use contact email OR smtp user as from (depends on provider policies, usually smtp user)
          if (!empty($dbSettings['contact_email']))
            $smtpConfig['from'] = $dbSettings['contact_email'];
          // Better to use smtp_user as 'from' to avoid spoofing rejections, but let's stick to user preference or default
          if (!empty($dbSettings['smtp_user']))
            $smtpConfig['from'] = $dbSettings['smtp_user'];

        }
      }
    } catch (Exception $e) {
      // Fallback to defaults
    }

    $mail = new \PHPMailer\PHPMailer\PHPMailer(true);

    try {
      //Server settings
      $mail->isSMTP();
      $mail->Host = $smtpConfig['host'];
      $mail->SMTPAuth = true;
      $mail->Username = $smtpConfig['user'];
      $mail->Password = $smtpConfig['pass'];
      $mail->SMTPSecure = \PHPMailer\PHPMailer\PHPMailer::ENCRYPTION_STARTTLS; // or ENCRYPTION_SMTPS
      $mail->Port = $smtpConfig['port'];
      $mail->CharSet = 'UTF-8';

      //Recipients
      $mail->setFrom($smtpConfig['from'], $smtpConfig['name']);
      $mail->addAddress($to);

      //Content
      $mail->isHTML($isHtml);
      $mail->Subject = $subject;
      $mail->Body = $body;
      $mail->AltBody = strip_tags($body);

      $mail->send();
      return true;
    } catch (\PHPMailer\PHPMailer\Exception $e) {
      error_log("Message could not be sent. Mailer Error: {$mail->ErrorInfo}");
      return false;
    }
  }

  public static function renderEmailView($view, $data = [])
  {
    // Inject Global Sytem Settings (Logo, Name, etc)
    if (file_exists(APP_PATH . '/models/Setting.php')) {
      require_once APP_PATH . '/models/Setting.php';
      if (class_exists('Setting')) {
        $settingModel = new Setting();
        $data['emailLogo'] = $settingModel->get('email_logo');
        $data['siteName'] = $settingModel->get('site_name', 'Cê Vai');
        $data['mainColor'] = $settingModel->get('color_brand_secondary', '#7000FF'); // Used for Buttons/Text
        $data['accentColor'] = $settingModel->get('color_brand_neon', '#00F3FF');   // Used for Accents
      }
    }

    // Extract data to variables
    extract($data);

    // Buffer basic content
    ob_start();
    $viewPath = APP_PATH . '/views/emails/' . $view . '.php';
    if (file_exists($viewPath)) {
      require $viewPath;
    } else {
      echo "View not found: $view";
    }
    $content = ob_get_clean();

    // Buffer layout with content
    ob_start();
    $layoutPath = APP_PATH . '/views/emails/layout.php';
    if (file_exists($layoutPath)) {
      require $layoutPath;
    } else {
      // Fallback if no layout
      echo $content;
    }
    return ob_get_clean();
  }


  public static function updateEnv($key, $value)
  {
    $path = APP_PATH . '/../.env';
    if (!file_exists($path)) {
      return false;
    }

    $lines = file($path);
    $newLines = [];
    $found = false;

    // Sanitize value (quote if space)
    if (strpos($value, ' ') !== false && strpos($value, '"') === false) {
      $value = '"' . $value . '"';
    }

    foreach ($lines as $line) {
      if (strpos(trim($line), $key . '=') === 0) {
        $newLines[] = "{$key}={$value}\n";
        $found = true;
      } else {
        $newLines[] = $line;
      }
    }

    if (!$found) {
      $newLines[] = "\n{$key}={$value}\n";
    }

    return file_put_contents($path, implode('', $newLines));
  }

  private static $siteNameCache = null;

  public static function getSiteName($default = 'Cê Vai')
  {
    if (self::$siteNameCache !== null) {
      return self::$siteNameCache;
    }

    $siteName = $default;
    try {
      if (file_exists(APP_PATH . '/models/Setting.php')) {
        require_once APP_PATH . '/models/Setting.php';
        if (class_exists('Setting')) {
          $settingModel = new Setting();
          $dbName = $settingModel->get('site_name');
          if (!empty($dbName)) {
            $siteName = $dbName;
          }
        }
      }
    } catch (Exception $e) {
      // Ignore DB errors, fallback to default
    }

    self::$siteNameCache = $siteName;
    return $siteName;
  }

  public static function csrf_token()
  {
    require_once APP_PATH . '/core/Csrf.php';
    return Csrf::getToken();
  }

  public static function csrf_field()
  {
    $token = self::csrf_token();
    return '<input type="hidden" name="csrf_token" value="' . $token . '">';
  }
}
