<?php

class UserTicket extends Model
{
  public function create($orderId, $userId, $ticketTypeId, $uniqueCode, $status = 'active')
  {
    $sql = "INSERT INTO user_tickets (order_id, user_id, ticket_type_id, unique_code, status, created_at) VALUES (:order_id, :user_id, :ticket_type_id, :unique_code, :status, NOW())";
    $stmt = $this->conn->prepare($sql);
    $stmt->bindValue(':order_id', $orderId);
    $stmt->bindValue(':user_id', $userId);
    $stmt->bindValue(':ticket_type_id', $ticketTypeId);
    $stmt->bindValue(':unique_code', $uniqueCode);
    $stmt->bindValue(':status', $status);

    return $stmt->execute();
  }

  public function countByTicketType($ticketTypeId)
  {
    $stmt = $this->conn->prepare("SELECT COUNT(*) FROM user_tickets WHERE ticket_type_id = :id");
    $stmt->bindValue(':id', $ticketTypeId);
    $stmt->execute();
    return $stmt->fetchColumn();
  }

  public function getByUserId($userId)
  {
    // Join with details for display
    $sql = "SELECT ut.*, tt.name as ticket_name, tt.batch_name, e.title as event_title, e.start_date, e.location 
                FROM user_tickets ut
                JOIN ticket_types tt ON ut.ticket_type_id = tt.id
                JOIN events e ON tt.event_id = e.id
                WHERE ut.user_id = :user_id
                ORDER BY ut.created_at DESC";
    $stmt = $this->conn->prepare($sql);
    $stmt->bindValue(':user_id', $userId);
    $stmt->execute();
    return $stmt->fetchAll();
  }

  public function findByCode($code)
  {
    $sql = "SELECT ut.*, u.name as user_name, u.email as user_email, 
                   tt.name as ticket_name, 
                   e.title as event_title, e.start_date, e.location, e.image_cover
              FROM user_tickets ut
              JOIN users u ON ut.user_id = u.id
              JOIN ticket_types tt ON ut.ticket_type_id = tt.id
              JOIN events e ON tt.event_id = e.id
              WHERE ut.unique_code = :code";
    $stmt = $this->conn->prepare($sql);
    $stmt->bindValue(':code', $code);
    $stmt->execute();
    return $stmt->fetch();
  }

  public function markAsUsed($id, $operatorId = null)
  {
    $sql = "UPDATE user_tickets SET status = 'used', used_at = NOW(), checked_in_by = :operator_id WHERE id = :id";
    $stmt = $this->conn->prepare($sql);
    $stmt->bindValue(':id', $id);
    $stmt->bindValue(':operator_id', $operatorId);
    return $stmt->execute();
  }



  public function updateStatusByOrder($orderId, $status)
  {
    $sql = "UPDATE user_tickets SET status = :status WHERE order_id = :order_id";
    $stmt = $this->conn->prepare($sql);
    $stmt->bindValue(':status', $status);
    $stmt->bindValue(':order_id', $orderId);
    return $stmt->execute();
  }

  public function activateByOrderId($orderId)
  {
    $sql = "UPDATE user_tickets SET status = 'active' WHERE order_id = :order_id";
    $stmt = $this->conn->prepare($sql);
    $stmt->bindValue(':order_id', $orderId);
    return $stmt->execute();
  }

  public function getCheckinStats($limit = 5, $filters = [])
  {
    $sql = "SELECT 
              e.title as event_title, 
              COUNT(ut.id) as total_sold,
              SUM(CASE WHEN ut.status = 'used' THEN 1 ELSE 0 END) as total_checked_in
            FROM user_tickets ut
            JOIN ticket_types tt ON ut.ticket_type_id = tt.id
            JOIN events e ON tt.event_id = e.id
            WHERE ut.status != 'cancelled'";

    // Filters
    if (!empty($filters['event_id'])) {
      $sql .= " AND e.id = :event_id";
    }
    if (!empty($filters['ticket_id'])) {
      $sql .= " AND tt.id = :ticket_id";
    }
    if (!empty($filters['start_date'])) {
      $sql .= " AND ut.created_at >= :start_date";
    }
    if (!empty($filters['end_date'])) {
      $sql .= " AND ut.created_at <= :end_date";
    }

    $sql .= " GROUP BY e.id ORDER BY total_sold DESC LIMIT :limit";

    $stmt = $this->conn->prepare($sql);
    $stmt->bindValue(':limit', (int) $limit, PDO::PARAM_INT);

    if (!empty($filters['event_id'])) {
      $stmt->bindValue(':event_id', $filters['event_id']);
    }
    if (!empty($filters['ticket_id'])) {
      $stmt->bindValue(':ticket_id', $filters['ticket_id']);
    }
    if (!empty($filters['start_date'])) {
      $stmt->bindValue(':start_date', $filters['start_date'] . ' 00:00:00');
    }
    if (!empty($filters['end_date'])) {
      $stmt->bindValue(':end_date', $filters['end_date'] . ' 23:59:59');
    }

    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
  }

  public function getAttendeesByEvent($eventId, $filters = [])
  {
    $sql = "SELECT ut.unique_code, ut.created_at as purchase_date, ut.status, ut.used_at,
                     u.name as user_name, u.email as user_email,
                     tt.name as ticket_name
              FROM user_tickets ut
              JOIN users u ON ut.user_id = u.id
              JOIN ticket_types tt ON ut.ticket_type_id = tt.id
              WHERE tt.event_id = :event_id";

    if (!empty($filters['ticket_id'])) {
      $sql .= " AND tt.id = :ticket_id";
    }

    $sql .= " ORDER BY u.name ASC";

    $stmt = $this->conn->prepare($sql);
    $stmt->bindValue(':event_id', $eventId);

    if (!empty($filters['ticket_id'])) {
      $stmt->bindValue(':ticket_id', $filters['ticket_id']);
    }

    $stmt->execute();
    return $stmt->fetchAll();
  }
  public function countByEventAndStatus($eventId, $status)
  {
    $sql = "SELECT COUNT(ut.id) 
            FROM user_tickets ut
            JOIN ticket_types tt ON ut.ticket_type_id = tt.id
            WHERE tt.event_id = :event_id AND ut.status = :status";
    $stmt = $this->conn->prepare($sql);
    $stmt->bindValue(':event_id', $eventId);
    $stmt->bindValue(':status', $status);
    $stmt->execute();
    return $stmt->fetchColumn();
  }
}
