<?php

/**
 * Database Updater Script (atualiza_db.php)
 * 
 * This script manages database schema changes and data updates.
 * Ideally run via CLI or a secured admin route.
 */

// Define path constants to load configuration
define('APP_PATH', __DIR__ . '/../app');
require_once APP_PATH . '/config/app_config.php';
require_once APP_PATH . '/config/database.php';

// Disable time limit for long migrations
set_time_limit(0);

echo "<h1>Atualizador de Banco de Dados</h1>";

try {
  // Connect to Database
  $database = new Database();
  $db = $database->connect();
  echo "<p style='color:green'>Conectado ao banco de dados com sucesso.</p>";
} catch (Exception $e) {
  die("<p style='color:red'>Erro ao conectar ao banco de dados: " . $e->getMessage() . "</p>");
}

// 1. Create Migrations Table
$sql = "CREATE TABLE IF NOT EXISTS system_migrations (
    id INT AUTO_INCREMENT PRIMARY KEY,
    version VARCHAR(255) NOT NULL UNIQUE,
    description TEXT,
    executed_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
)";

try {
  $db->exec($sql);
  echo "<p>Tabela de migrações verificada.</p>";
} catch (Exception $e) {
  die("<p style='color:red'>Erro ao verificar tabela de migrações: " . $e->getMessage() . "</p>");
}

// 2. Define Updates (Versions)
// Format: 'version_id' => ['description' => '...', 'sql' => '...' or function($db)]
$updates = [
  '1.0.0' => [
    'description' => 'Initial Base System Check',
    'callback' => function ($db) {
      // Just a placeholder to mark the baseline
      return true;
    }
  ],
  '1.0.1' => [
    'description' => 'Ensure users table has asaas_customer_id',
    'sql' => "SHOW COLUMNS FROM users LIKE 'asaas_customer_id'",
    'callback' => function ($db) {
      // Check if column exists, if not add it
      $stmt = $db->query("SHOW COLUMNS FROM users LIKE 'asaas_customer_id'");
      if ($stmt->rowCount() == 0) {
        $db->exec("ALTER TABLE users ADD COLUMN asaas_customer_id VARCHAR(255) DEFAULT NULL AFTER phone");
        return "Coluna asaas_customer_id adicionada.";
      }
      return "Coluna asaas_customer_id já existe.";
    }
  ],
  '1.0.2' => [
    'description' => 'Ensure orders table has external_reference',
    'callback' => function ($db) {
      $stmt = $db->query("SHOW COLUMNS FROM orders LIKE 'external_reference'");
      if ($stmt->rowCount() == 0) {
        $db->exec("ALTER TABLE orders ADD COLUMN external_reference VARCHAR(255) DEFAULT NULL");
        return "Coluna external_reference adicionada.";
      }
      return "Coluna external_reference já existe.";
    }
  ],
  '1.0.3' => [
    'description' => 'Fix ENUM status in orders if needed (checking for refunding status)',
    'callback' => function ($db) {
      // This is complex in MySQL to check ENUM values purely, so we just try to modify it to include all needed statuses safely
      // Just a safeguard update
      // Note: ALTER TABLE on large tables can be slow.
      // here we assume it handles 'pending','paid','cancelled','refunded','failed'
      // We won't force it unless we know it's missing, to avoid locks.
      return "Verificação de status pulada para evitar bloqueios desnecessários.";
    }
  ]
];

// 3. Run Updates
$executedCount = 0;

echo "<ul>";

foreach ($updates as $version => $update) {
  // Check if executed
  $stmt = $db->prepare("SELECT id FROM system_migrations WHERE version = ?");
  $stmt->execute([$version]);
  if ($stmt->fetch()) {
    continue; // Already executed
  }

  echo "<li><strong>Executando atualização $version:</strong> " . $update['description'] . "... ";

  try {
    $db->beginTransaction();

    $resultMessage = "";

    // Execute SQL string if present
    if (isset($update['sql']) && is_string($update['sql'])) {
      // If it's a simple SQL (not used strictly in this structure usually, but supported)
      // Actually, for conditional schema changes, generic SQL string often fails if column exists.
      // So we prefer callbacks. But if provided:
      // $db->exec($update['sql']); 
      // We won't use raw SQL in this example structure to prefer the 'callback' flexibility.
    }

    // Execute Callback if present [Preferred]
    if (isset($update['callback']) && is_callable($update['callback'])) {
      $res = $update['callback']($db);
      if (is_string($res)) {
        $resultMessage = $res;
      }
    }

    // Record Migration
    $stmtOpts = $db->prepare("INSERT INTO system_migrations (version, description) VALUES (?, ?)");
    $stmtOpts->execute([$version, $update['description']]);

    $db->commit();
    echo "<span style='color:green'>Sucesso. $resultMessage</span></li>";
    $executedCount++;

  } catch (Exception $e) {
    $db->rollBack();
    echo "<span style='color:red'>Falha! " . $e->getMessage() . "</span></li>";
    echo "</ul>";
    die("<h3>Atualização interrompida devido a erro crítico.</h3>");
  }
}

echo "</ul>";

if ($executedCount === 0) {
  echo "<p style='color:blue'>O sistema já está atualizado. Nenhuma alteração pendente.</p>";
} else {
  echo "<p style='color:green'><strong>$executedCount atualização(ões) aplicada(s) com sucesso.</strong></p>";
}

echo "<hr><p><a href='/'>Voltar para a Home</a></p>";
