<?php

// PHP Built-in Server Routing Compatibility
if (php_sapi_name() === 'cli-server') {
  $url = parse_url($_SERVER['REQUEST_URI']);
  $file = __DIR__ . $url['path'];
  if (is_file($file)) {
    return false; // Serve static file directly
  }
}

// Secure Session Cookie Settings
ini_set('session.cookie_httponly', 1);
ini_set('session.use_only_cookies', 1);
ini_set('session.use_strict_mode', 1);

// Basic domain extraction without port
$domain = $_SERVER['HTTP_HOST'] ?? 'localhost';
if (strpos($domain, ':') !== false) {
  $domain = explode(':', $domain)[0];
}

session_set_cookie_params([
  'lifetime' => 86400,
  'path' => '/',
  'domain' => $domain == 'localhost' ? '' : $domain, // Empty for localhost to avoid issues
  'secure' => isset($_SERVER['HTTPS']), // True only if HTTPS
  'httponly' => true,
  'samesite' => 'Strict'
]);

session_start();

// Validar se o composer foi instalado
if (file_exists(dirname(__DIR__) . '/vendor/autoload.php')) {
  require_once dirname(__DIR__) . '/vendor/autoload.php';
}

require_once __DIR__ . '/../app/config/app_config.php';
require_once APP_PATH . '/core/Router.php';
require_once APP_PATH . '/core/Controller.php';
require_once APP_PATH . '/core/Model.php';
require_once APP_PATH . '/core/Helpers.php';

// Instantiate Router
$router = new Router();

// --- Define Routes ---

// Public Routes
$router->get('/', 'HomeController@index');
$router->get('/terms', function () {
  require APP_PATH . '/views/pages/terms.php';
});
$router->get('/privacy', function () {
  require APP_PATH . '/views/pages/privacy.php';
});
$router->get('/contact', 'HomeController@contact');
$router->post('/contact/process', 'HomeController@processContact');
$router->get('/politica-reembolso', 'HomeController@refundPolicy');
$router->get('/agenda', 'HomeController@list');
$router->get('/event/(\S+)', 'EventController@show'); // Regex for slug
$router->get('/checkout', 'CheckoutController@index');
$router->get('/checkout/pay', 'CheckoutController@pay');
$router->get('/checkout/payment/(\d+)', 'CheckoutController@payment');
$router->post('/checkout/process', 'CheckoutController@process');
$router->get('/login', 'AuthController@login');
$router->post('/login', 'AuthController@authenticate');
$router->get('/register', 'AuthController@register');
$router->post('/register', 'AuthController@store');
$router->get('/logout', 'AuthController@logout');
$router->get('/forgot-password', 'AuthController@forgotPassword');
$router->post('/forgot-password', 'AuthController@sendResetLink');
$router->get('/reset-password/(\S+)', 'AuthController@resetPassword');
$router->post('/reset-password', 'AuthController@updatePassword');
$router->get('/activate/(\S+)', 'AuthController@activate');

// Operators ManagementRoutes
$router->get('/admin/operators', 'AdminController@operators');
$router->post('/admin/operators/store', 'AdminController@storeOperator');
$router->post('/admin/operators/update', 'AdminController@updateOperator');
$router->get('/admin/operators/delete', 'AdminController@deleteOperator');
$router->get('/admin/events/operators', 'AdminController@manageEventOperators');
$router->post('/admin/events/operators/add', 'AdminController@addEventOperator');
$router->get('/admin/events/operators/remove', 'AdminController@removeEventOperator');

// Operator Dashboard (New Role)
$router->get('/operator/dashboard', 'OperatorController@dashboard');
$router->get('/operator/event/(\d+)/checkin', 'OperatorController@checkin');
$router->post('/operator/checkin/process', 'OperatorController@processCheckin');

// Client Routes
$router->get('/client/dashboard', 'ClientController@dashboard');
$router->get('/client/tickets', 'ClientController@tickets');
$router->get('/client/profile', 'ClientController@profile');
$router->post('/client/profile/update', 'ClientController@updateProfile');
$router->get('/client/tickets/([\w-]+)/view', 'ClientController@viewTicket');

// Admin Routes
$router->get('/admin', function () {
  header('Location: /admin/dashboard');
  exit;
});
$router->get('/admin/login', 'AdminAuthController@login');
$router->post('/admin/login', 'AdminAuthController@authenticate');
$router->get('/admin/dashboard', 'AdminController@dashboard');
$router->get('/admin/events', 'AdminController@events');
$router->get('/admin/events/create', 'AdminController@createEvent');
$router->post('/admin/events/store', 'AdminController@storeEvent');
$router->get('/admin/events/tickets', 'AdminController@manageTickets');
$router->post('/admin/events/tickets/store', 'AdminController@storeTicket');
$router->get('/admin/events/tickets/edit', 'AdminController@editTicket');
$router->post('/admin/events/tickets/update', 'AdminController@updateTicket');
$router->get('/admin/checkin', 'AdminController@checkin');
$router->post('/admin/checkin/process', 'AdminController@processCheckin');
$router->get('/admin/events/edit', 'AdminController@editEvent');
$router->post('/admin/events/update', 'AdminController@updateEvent');
$router->get('/admin/events/highlight', 'AdminController@setHighlight');
$router->get('/admin/events/speakers', 'AdminController@manageSpeakers');
$router->post('/admin/events/speakers/store', 'AdminController@storeSpeaker');
$router->get('/admin/events/speakers/delete', 'AdminController@deleteSpeaker');
$router->get('/admin/clients', 'AdminController@clients');
$router->get('/admin/clients/edit', 'AdminController@editClient');
$router->post('/admin/clients/update', 'AdminController@updateClient');
$router->get('/admin/clients/delete', 'AdminController@deleteClient');
$router->get('/admin/reports', 'AdminController@reports');
$router->get('/admin/reports/export', 'AdminController@exportReport');
$router->get('/admin/settings', 'AdminController@settings');
$router->post('/admin/settings/update', 'AdminController@updateSettings');
$router->post('/admin/settings/test-email', 'AdminController@testEmail');
$router->get('/admin/order', 'AdminController@order');
$router->get('/admin/orders/create', 'AdminController@createOrder');
$router->post('/admin/orders/store', 'AdminController@storeManualOrder');

$router->get('/admin/orders', 'AdminController@orders');
$router->get('/admin/orders/edit', 'AdminController@editOrder');
$router->post('/admin/orders/update', 'AdminController@updateOrder');
$router->post('/admin/order/resend_email', 'AdminController@resendOrderEmail');

$router->get('/admin/events/remove', 'AdminController@deleteEvent');
$router->get('/admin/events/tickets/remove', 'AdminController@deleteTicket');

$router->post('/api/webhook/asaas', 'WebhookController@handle');
$router->get('/api/webhook/asaas', function () {
  echo "Webhook endpoint is active. Use POST to send data.";
});

// Run Router
try {
  $router->resolve();
} catch (Exception $e) {
  error_log($e->getMessage());
  http_response_code(500);
  require APP_PATH . '/views/errors/500.php';
}
